<?php

namespace App\Http\Controllers\Store;

use App\Http\Controllers\Controller;
use App\Models\Coupon;
use App\Models\Package;
use App\Models\StoreAction;
use App\Models\Server;
use App\Models\Transaction;
use App\Services\PayPalService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Stripe\Stripe;
use Stripe\Checkout\Session as StripeSession;
use Stripe\Webhook;

class PaymentController extends Controller
{
    protected $paypalService;

    public function __construct(PayPalService $paypalService)
    {
        $this->paypalService = $paypalService;
        
        // Initialize Stripe if enabled
        if (config_value('store_stripe_enabled', false)) {
            Stripe::setApiKey(config_value('store_stripe_secret'));
        }
    }

    /**
     * Show checkout page
     */
    public function checkout($slug, Request $request)
    {
        $serverId = $request->get('server_id') ?? session('current_server_id');
        $server = null;
        
        if ($serverId) {
            $server = Server::where('is_active', true)->find($serverId);
        }
        
        $query = Package::where('slug', $slug)
            ->where('is_active', true);
            
        // If server is specified, ensure package is available on that server
        if ($server) {
            $query->forServer($server->id);
        } else {
            // If no server specified, ensure package is assigned to at least one server
            $query->withServers();
        }
        
        $package = $query->firstOrFail();
            
        // Get coupon from session
        $couponData = Session::get('coupon');
        
        // Check which payment methods are enabled
        $paymentMethods = [
            'paypal' => config_value('store_paypal_enabled', true) && $this->paypalService->isConfigured(),
            'stripe' => config_value('store_stripe_enabled', false) && config_value('store_stripe_secret')
        ];
        
        // If no payment methods are enabled, throw an error
        if (!array_filter($paymentMethods)) {
            abort(503, 'No payment methods are currently available');
        }
        
        return view('store.checkout', compact('package', 'couponData', 'server', 'paymentMethods'));
    }
    
    /**
     * Process payment - Only handles Stripe now (PayPal goes through API)
     */
    public function processPayment(Request $request, $slug)
    {
        // Only Stripe payments go through this route now
        
        $request->validate([
            'payment_method' => 'required|in:stripe'
        ]);

        if (!config_value('store_stripe_enabled', false)) {
            return redirect()->back()->with('error', 'Stripe payments are currently disabled.');
        }
        
        $serverId = $request->get('server_id') ?? session('current_server_id');
        $server = null;
        
        if ($serverId) {
            $server = Server::where('is_active', true)->find($serverId);
        }
        
        $query = Package::where('slug', $slug)
            ->where('is_active', true);
            
        // If server is specified, ensure package is available on that server
        if ($server) {
            $query->forServer($server->id);
        } else {
            // If no server specified, ensure package is assigned to at least one server
            $query->withServers();
        }
        
        $package = $query->firstOrFail();
            
        // Get coupon from session and apply if valid
        $couponData = Session::get('coupon');
        $couponId = null;
        $discountAmount = 0;
        $originalAmount = $package->price;
        $finalAmount = $package->price;
        
        if ($couponData) {
            $coupon = Coupon::find($couponData['id']);
            
            if ($coupon && $coupon->isValid()) {
                $couponId = $coupon->id;
                $discountAmount = $couponData['discount_amount'];
                $finalAmount = $couponData['final_price'];
                
                // Increment coupon uses
                $coupon->incrementUses();
            }
        }
        
        // Create a pending transaction
        $transaction = Transaction::create([
            'user_id' => Auth::id(),
            'package_id' => $package->id,
            'server_id' => $server ? $server->id : null,
            'amount' => $finalAmount,
            'original_amount' => $originalAmount,
            'discount_amount' => $discountAmount,
            'coupon_id' => $couponId,
            'payment_method' => 'stripe',
            'status' => 'pending',
            'ip_address' => $request->ip(),
        ]);
        
        // Clear coupon from session after use
        Session::forget('coupon');
        
        // Process Stripe payment
        return $this->processStripePayment($transaction, $package, $server);
    }

    /**
     * Create PayPal order (API endpoint)
     */
    public function createPayPalOrder(Request $request)
    {
        try {
            $request->validate([
                'package_slug' => 'required|string',
                'server_id' => 'nullable|integer'
            ]);

            $serverId = $request->get('server_id');
            $server = null;
            
            if ($serverId) {
                $server = Server::where('is_active', true)->find($serverId);
            }
            
            $query = Package::where('slug', $request->package_slug)
                ->where('is_active', true);
                
            if ($server) {
                $query->forServer($server->id);
            } else {
                $query->withServers();
            }
            
            $package = $query->firstOrFail();
            
            // Get coupon from session and apply if valid
            $couponData = Session::get('coupon');
            $couponId = null;
            $discountAmount = 0;
            $originalAmount = $package->price;
            $finalAmount = $package->price;
            
            if ($couponData) {
                $coupon = Coupon::find($couponData['id']);
                
                if ($coupon && $coupon->isValid()) {
                    $couponId = $coupon->id;
                    $discountAmount = $couponData['discount_amount'];
                    $finalAmount = $couponData['final_price'];
                    
                    // Increment coupon uses
                    $coupon->incrementUses();
                }
            }
            
            // Create a pending transaction
            $transaction = Transaction::create([
                'user_id' => Auth::id(),
                'package_id' => $package->id,
                'server_id' => $server ? $server->id : null,
                'amount' => $finalAmount,
                'original_amount' => $originalAmount,
                'discount_amount' => $discountAmount,
                'coupon_id' => $couponId,
                'payment_method' => 'paypal',
                'status' => 'pending',
                'ip_address' => $request->ip(),
            ]);
            
            // Clear coupon from session after use
            Session::forget('coupon');
            
            // Create PayPal order using the actual service method signature
            $itemName = $package->name . ' - ' . config_value('site_name', 'EdgeRP');
            if ($server) {
                $itemName .= ' (' . $server->name . ')';
            }

            $paypalOrder = $this->paypalService->createOrder($transaction, $itemName, $package->description);
            
            if (!$paypalOrder || !isset($paypalOrder['id'])) {
                Log::error('PayPal order creation failed', [
                    'response' => $paypalOrder,
                    'transaction_id' => $transaction->id
                ]);
                return response()->json(['error' => 'Failed to create PayPal order'], 500);
            }
            
            // Store PayPal order ID in transaction
            $transaction->update(['paypal_order_id' => $paypalOrder['id']]);
            
            Log::info('PayPal order created successfully', [
                'paypal_order_id' => $paypalOrder['id'],
                'transaction_id' => $transaction->id,
                'user_id' => Auth::id()
            ]);
            
            return response()->json([
                'id' => $paypalOrder['id'],
                'transaction_id' => $transaction->id
            ]);
            
        } catch (\Exception $e) {
            Log::error('PayPal create order error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'user_id' => Auth::id(),
                'request_data' => $request->all()
            ]);
            
            return response()->json(['error' => 'Payment processing failed'], 500);
        }
    }

    /**
     * Process Stripe payment
     */
    private function processStripePayment($transaction, $package, $server = null)
    {
        try {
            $itemName = $package->name . ' - ' . config_value('site_name', 'EdgeRP');
            if ($server) {
                $itemName .= ' (' . $server->name . ')';
            }

            $stripeSession = StripeSession::create([
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => strtolower(config_value('store_currency', 'GBP')),
                        'product_data' => [
                            'name' => $itemName,
                            'description' => $package->description,
                        ],
                        'unit_amount' => intval($transaction->amount * 100), // Convert to pennies (cents for you americans)
                    ],
                    'quantity' => 1,
                ]],
                'mode' => 'payment',
                'success_url' => route('stripe.success', $transaction->id) . '?session_id={CHECKOUT_SESSION_ID}',
                'cancel_url' => route('payment.cancel', $transaction->id),
                'metadata' => [
                    'transaction_id' => $transaction->id,
                    'user_id' => Auth::id(),
                    'package_id' => $package->id,
                    'server_id' => $server ? $server->id : null,
                ],
                'customer_email' => Auth::user()->email ?? null,
            ]);

            // Store Stripe session ID in transaction
            $transaction->update(['stripe_session_id' => $stripeSession->id]);

            return redirect($stripeSession->url);

        } catch (\Exception $e) {
            Log::error('Stripe payment creation failed: ' . $e->getMessage());
            return redirect()->route('store.payment.checkout', $package->slug)
                ->with('error', 'Payment processing failed. Please try again.');
        }
    }

    /**
     * Handle PayPal success (supports both order IDs and transaction IDs)
     */
    public function paypalSuccess($orderIdOrTransactionId, Request $request)
    {
        $token = $request->get('token');
        $payerId = $request->get('PayerID');
        
        // Check if this is a PayPal order ID (new flow) or transaction ID (old flow)
        if ($token && $token === $orderIdOrTransactionId) {
            // New flow: orderIdOrTransactionId is actually the PayPal order ID
            $paypalOrderId = $orderIdOrTransactionId;
            
            // Find transaction by PayPal order ID
            $transaction = Transaction::where('paypal_order_id', $paypalOrderId)->first();
            
            if (!$transaction) {
                Log::error('Transaction not found for PayPal order ID: ' . $paypalOrderId);
                return redirect()->route('store.index')->with('error', 'Transaction not found.');
            }
        } else {
            // Old flow: orderIdOrTransactionId is the transaction ID
            $transaction = Transaction::findOrFail($orderIdOrTransactionId);
        }

        if ($transaction->user_id !== Auth::id()) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        // If already completed, just show success
        if ($transaction->status === 'completed') {
            return view('store.success', compact('transaction'));
        }

        try {
            // Capture the PayPal order using the service
            $captureResult = $this->paypalService->captureOrder($transaction->paypal_order_id);
            
            if ($captureResult && $captureResult['status'] === 'COMPLETED') {
                // Update transaction
                $transaction->update([
                    'status' => 'completed',
                    'transaction_id' => $captureResult['id'],
                    'paypal_data' => $captureResult,
                ]);

                // Process package benefits
                $this->processPackageBenefits($transaction);
                
                Log::info('PayPal payment completed successfully', [
                    'transaction_id' => $transaction->id,
                    'paypal_capture_id' => $captureResult['id'],
                    'user_id' => $transaction->user_id
                ]);
            } else {
                $transaction->update(['status' => 'failed']);
                Log::warning('PayPal payment capture failed or incomplete', [
                    'transaction_id' => $transaction->id,
                    'capture_status' => $captureResult['status'] ?? 'unknown'
                ]);
                
                return redirect()->route('store.index')->with('error', 'Payment could not be completed.');
            }
        } catch (\Exception $e) {
            Log::error('PayPal capture failed: ' . $e->getMessage());
            $transaction->update(['status' => 'failed']);
            
            return redirect()->route('store.index')->with('error', 'Payment processing failed.');
        }
        
        return view('store.success', compact('transaction'));
    }
    
    /**
     * Handle Stripe success
     */
    public function stripeSuccess($transactionId, Request $request)
    {
        $sessionId = $request->get('session_id');
        $transaction = Transaction::findOrFail($transactionId);

        if ($transaction->user_id !== Auth::id()) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        try {
            // Retrieve the Stripe session
            $stripeSession = StripeSession::retrieve($sessionId);
            
            if ($stripeSession->payment_status === 'paid') {
                // Update transaction
                $transaction->update([
                    'status' => 'completed',
                    'transaction_id' => $stripeSession->payment_intent,
                    'stripe_data' => [
                        'session_id' => $sessionId,
                        'payment_intent' => $stripeSession->payment_intent,
                        'amount_total' => $stripeSession->amount_total,
                        'currency' => $stripeSession->currency,
                    ]
                ]);

                // Process package benefits
                $this->processPackageBenefits($transaction);
            }
        } catch (\Exception $e) {
            Log::error('Stripe success processing failed: ' . $e->getMessage());
        }
        
        return view('store.success', compact('transaction'));
    }

    /**
     * Handle Stripe webhook
     */
    public function stripeWebhook(Request $request)
    {
        $payload = $request->getContent();
        $sigHeader = $request->header('Stripe-Signature');
        $endpointSecret = config_value('store_stripe_webhook_secret', '');

        try {
            $event = Webhook::constructEvent(
                $payload, $sigHeader, $endpointSecret
            );
        } catch (\UnexpectedValueException $e) {
            Log::error('Invalid Stripe webhook payload');
            return response('Invalid payload', 400);
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            Log::error('Invalid Stripe webhook signature');
            return response('Invalid signature', 400);
        }

        // Handle the event
        switch ($event->type) {
            case 'checkout.session.completed':
                $session = $event->data->object;
                $this->handleStripePaymentSuccess($session);
                break;
            case 'checkout.session.expired':
                $session = $event->data->object;
                $this->handleStripePaymentExpired($session);
                break;
            default:
                Log::info('Received unknown Stripe event type: ' . $event->type);
        }

        return response('OK', 200);
    }

    /**
     * Handle PayPal webhook
     */
    public function paypalWebhook(Request $request)
    {
        try {
            $headers = $request->headers->all();
            $body = $request->getContent();
            
            // Verify the webhook signature using the service
            if (!$this->paypalService->verifyWebhookSignature($headers, $body)) {
                Log::error('PayPal webhook signature verification failed');
                return response('Forbidden', 403);
            }
            
            $data = json_decode($body, true);
            
            if (isset($data['event_type'])) {
                switch ($data['event_type']) {
                    case 'CHECKOUT.ORDER.APPROVED':
                    case 'PAYMENT.CAPTURE.COMPLETED':
                        $this->handlePayPalPaymentCompleted($data);
                        break;
                    case 'PAYMENT.CAPTURE.DENIED':
                    case 'CHECKOUT.ORDER.CANCELLED':
                        $this->handlePayPalPaymentFailed($data);
                        break;
                    default:
                        Log::info('Received unknown PayPal webhook event type: ' . $data['event_type']);
                }
            }
            
            return response('OK', 200);
            
        } catch (\Exception $e) {
            Log::error('PayPal webhook error: ' . $e->getMessage());
            return response('OK', 200);
        }
    }

    /**
     * Handle Stripe payment success from webhook
     */
    private function handleStripePaymentSuccess($session)
    {
        $transactionId = $session->metadata->transaction_id ?? null;
        
        if ($transactionId) {
            $transaction = Transaction::find($transactionId);
            
            if ($transaction && $transaction->status === 'pending') {
                $transaction->update([
                    'status' => 'completed',
                    'transaction_id' => $session->payment_intent,
                    'stripe_data' => [
                        'session_id' => $session->id,
                        'payment_intent' => $session->payment_intent,
                        'amount_total' => $session->amount_total,
                        'currency' => $session->currency,
                    ]
                ]);

                $this->processPackageBenefits($transaction);
            }
        }
    }

    /**
     * Handle Stripe payment expiration from webhook
     */
    private function handleStripePaymentExpired($session)
    {
        $transactionId = $session->metadata->transaction_id ?? null;
        
        if ($transactionId) {
            $transaction = Transaction::find($transactionId);
            
            if ($transaction && $transaction->status === 'pending') {
                $transaction->update(['status' => 'expired']);
            }
        }
    }
    
    /**
     * Handle PayPal payment completion from webhook
     */
    private function handlePayPalPaymentCompleted($data)
    {
        $resource = $data['resource'] ?? [];
        $customId = $resource['custom_id'] ?? null;
        
        if ($customId) {
            $transaction = Transaction::find($customId);
            
            if ($transaction && $transaction->status === 'pending') {
                $transaction->update([
                    'status' => 'completed',
                    'transaction_id' => $resource['id'] ?? null,
                    'paypal_data' => $data,
                ]);

                $this->processPackageBenefits($transaction);
                
                Log::info('PayPal webhook payment completed', [
                    'transaction_id' => $transaction->id,
                    'event_type' => $data['event_type'],
                ]);
            }
        }
    }
    
    /**
     * Handle PayPal payment failure from webhook
     */
    private function handlePayPalPaymentFailed($data)
    {
        $resource = $data['resource'] ?? [];
        $customId = $resource['custom_id'] ?? null;
        
        if ($customId) {
            $transaction = Transaction::find($customId);
            
            if ($transaction && $transaction->status === 'pending') {
                $transaction->update(['status' => 'failed']);
                
                Log::info('PayPal webhook payment failed', [
                    'transaction_id' => $transaction->id,
                    'event_type' => $data['event_type'],
                ]);
            }
        }
    }
    
    /**
     * Generic success page
     */
    public function success($transactionId)
    {
        $transaction = Transaction::findOrFail($transactionId);

        if ($transaction->user_id !== Auth::id()) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        return view('store.success', compact('transaction'));
    }
    
    /**
     * Generic cancel page
     */
    public function cancel($transactionId)
    {
        $transaction = Transaction::findOrFail($transactionId);
        $transaction->update(['status' => 'cancelled']);

        if ($transaction->user_id !== Auth::id()) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        return view('store.cancel', compact('transaction'));
    }
    
    /**
     * Process package benefits and create store actions
     */
    private function processPackageBenefits($transaction)
    {
        $package = $transaction->package;
        $user = $transaction->user;
        $server = $transaction->server;
        
        $user->unlock(new \App\Achievements\StorePurchase());
        
        // Prepare actions array based on package type and actions
        $actions = [];
        
        // If package has defined actions, use those
        if ($package->actions && is_array($package->actions)) {
            foreach ($package->actions as $action) {
                $actionData = [
                    'type' => $action['type']
                ];
                
                switch ($action['type']) {
                    case 'rank':
                        $actionData['rank'] = $action['value'];
                        $actionData['duration'] = $action['duration'] ?? 0;
                        break;
                        
                    case 'money':
                        $actionData['amount'] = floatval($action['value']);
                        break;
                        
                    case 'custom_lua':
                        $actionData['lua_code'] = $action['value'];
                        break;
                        
                    case 'console_command':
                        $actionData['command'] = $action['value'];
                        break;
                }
                
                $actions[] = $actionData;
            }
        } else {
            // Fallback to package type for backwards compatibility
            switch ($package->type) {
                case 'rank':
                    $actions[] = [
                        'type' => 'rank',
                        'rank' => $package->rank ?? 'vip',
                        'duration' => $package->duration ?? 0,
                    ];
                    break;
                    
                case 'money':
                    $actions[] = [
                        'type' => 'money',
                        'amount' => $package->money_amount ?? 0,
                    ];
                    break;
                    
                case 'custom_lua':
                    $actions[] = [
                        'type' => 'custom_lua',
                        'lua_code' => $package->lua_code ?? '',
                    ];
                    break;
                    
                case 'console_command':
                    $actions[] = [
                        'type' => 'console_command',
                        'command' => $package->command ?? '',
                    ];
                    break;
                    
                case 'combo':
                    // Handle combo packages
                    $packageData = $package->data ?? [];
                    
                    if (isset($packageData['money'])) {
                        $actions[] = [
                            'type' => 'money',
                            'amount' => $packageData['money'],
                        ];
                    }
                    
                    if (isset($packageData['rank'])) {
                        $actions[] = [
                            'type' => 'rank',
                            'rank' => $packageData['rank']['name'] ?? 'vip',
                            'duration' => $packageData['rank']['duration'] ?? 0,
                        ];
                    }
                    
                    if (isset($packageData['custom_lua'])) {
                        $actions[] = [
                            'type' => 'custom_lua',
                            'lua_code' => $packageData['custom_lua'],
                        ];
                    }
                    
                    if (isset($packageData['console_commands'])) {
                        foreach ($packageData['console_commands'] as $command) {
                            $actions[] = [
                                'type' => 'console_command',
                                'command' => $command,
                            ];
                        }
                    }
                    break;
                    
                default:
                    Log::warning("Unknown package type: {$package->type} for package {$package->name}");
                    break;
            }
        }
        
        // If there are actions to process, store them in the store_actions table
        if (!empty($actions)) {
            $serverIdToUse = $server ? $server->id : null;
            if (!$serverIdToUse) {
                $defaultServer = \App\Models\Server::where('is_active', true)->first();
                $serverIdToUse = $defaultServer ? $defaultServer->id : null;
            }
            
            \App\Models\StoreAction::create([
                'user_id' => $user->id,
                'server_id' => $serverIdToUse,
                'transaction_id' => $transaction->id,
                'package_id' => $package->id,
                'actions' => $actions,
                'executed' => false,
                'received_at' => null,
            ]);
        }
        
        Log::info("Created store action for user {$user->name} (SteamID: {$user->steamid}), package {$package->name}" . 
                  ($server ? " on server {$server->name}" : "") . 
                  " with " . count($actions) . " actions");
    }

    /**
     * Get PayPal order details (API endpoint)
     */
    public function getOrderDetails($orderId)
    {
        try {
            $orderDetails = $this->paypalService->getOrderDetails($orderId);
            
            if (!$orderDetails) {
                return response()->json(['error' => 'Order not found'], 404);
            }
            
            return response()->json($orderDetails);
            
        } catch (\Exception $e) {
            Log::error('PayPal get order details error: ' . $e->getMessage(), [
                'order_id' => $orderId
            ]);
            
            return response()->json(['error' => 'Failed to get order details'], 500);
        }
    }

    /**
     * Check transaction status (API endpoint)
     */
    public function checkStatus($id)
    {
        
        $transaction = Transaction::findOrFail($id);

        if ($transaction->user_id !== Auth::id()) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        return response()->json([
            'status' => $transaction->status,
            'transaction_id' => $transaction->transaction_id
        ]);
    }
}